/*

	Pegg 0.15 - A small printing program for the
	Casio "EGG" KP-C10/C50/KL-P1000 label Printer.

	Copyright (c) 2003,2004 Daniel Amkreutz, <daniel.amkreutz@tu-harburg.de>

        IMPORTANT NOTICE:

	* This software was designed using data made available by and is released
          with the permission of CASIO COMPUTER CO., LTD.
       * CASIO COMPUTER Co., LTD. assumes no responsibility for the content of
          this software.
       * Please do not contact CASIO COMPUTER CO., LTD. with any inquiries
          concerning this software.

	A large amount of code is borrowed from the libusb Documentation.
	The communication API of the Printer has been greatly documented
	and published by CASIO COMPUTER CO., LTD

							Many Thanks.

        This program is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

	This program is free software; you can redistribute it and/or modify
        it under the terms of the GNU General Public License as published by
        the Free Software Foundation; either version 2 of the License, or
        (at your option) any later version.

	You may use this program for whatever you want as long the above written
	text remains.

	FOR KP-C50:

	If you want to test pegg with the KP-C50 please modify line 209 as
	described in this line


	Please support Pegg by testing it and reporting bugs. !
*/

#include <strings.h>
#include <stdio.h>
#include <usb.h>

int _interfacenumber, 	/* Interface and Endpoints of the printer*/
    _endpoint_r,
    _endpoint_w,
    density = 3;	/* Set the default density */

char _read_buffer[8],	/* Read and Write buffer */
     _write_buffer[64],
     data[4095];	/* Buffer for Datafile */

char ESC = 0x1b,	/* The Printer's contol codes*/
     STX = 0x02,
     EOT = 0x04,	/* Reffer to the Command Scpec's by Casio */
     ENQ = 0x05,
     ACK = 0x06,
     NAK = 0x15,
     SYN = 0x16,
     CAN = 0x18,
     EOS = 0x1e,
     WIT = 0x1f;

char _B = 0x42, 	/* The Printer's command codes I*/
     _F = 0x46,
     _G = 0x47,		/* Reffer to the Command Scpec's by Casio */
     _I = 0x49,
     _N = 0x4e,
     _P = 0x50,
     _T = 0x54;

struct usb_bus *busses;
struct usb_bus *bus;
struct usb_device *dev;
usb_dev_handle *_eggprinter;


int main (int argc, char *argv[]) {	/* The Main Program handles commandline args and processes the image-file */

  int test=0,
      c=0;

  opterr = 0;
  bzero(data, 4096);

  if (argc < 2) {
    printf("Pegg Version 0.15 - (c)2003,04 Daniel Amkreutz:\n\tWrong number of arguments.\n\n	Usage:   pegg (-1,2,3,4,5) (-t) RAW-FILE\n\n	Options:\n	-1,2,3,4,5	set print density 0=light 5=dark default=3\n	-t		print Information about device\n\n\n");
    exit(0);
  }
   printf("Pegg Version 0.15 - (c)2003,04 Daniel Amkreutz\n");
 while ((c = getopt (argc, argv, "12345t")) != -1) {
    switch (c) {
      case 't':
        test = 1;			/* Handle commandline parameters */
        break;
      case '1':
        density = 1;			/* -1,2,3,4,5 = density */
        break;
      case '2':
        density = 2;
        break;
      case '3':
        density = 3;
        break;
      case '4':
        density = 4;
        break;
      case '5':
        density = 5;
        break;
    }
  }

  if(test == 1) {
   display_printer_status();
   exit(0);
  }

  if(argv[optind]) {

  FILE * raw_file;
  char c;
  int n = 0;

  raw_file = fopen(argv[optind],"rb");

  if (raw_file==NULL){
    printf("Error opening file: %s !\n", argv[optind]);
    exit(-1);
  } else {
    for (n=0; n < 4096; n++) {
      c = getc (raw_file);
      data[n] = c;
    }
    fclose (raw_file);
    printf ("%i Bytes read.\n",n);
  }
  send_raw_data();
  }else {

  printf("No Raw-File specified. \n");
  }
  exit(0);
}


int write_to_printer(int bytes) {	/* Function to write the date in _write_buffer to the printer */

  					/* Printer accepts only Packets site of 1,2,16,64 bytes */

  if (bytes == 1 || bytes == 2 || bytes == 16 || bytes == 64){

    if (usb_bulk_write(_eggprinter, _endpoint_w, (void *)_write_buffer ,bytes ,500) == bytes) {
      return 0;
    }
    return -2;			/* Error code -1 = wrong size */
  }
  return -1;			/* Error code -2 = transmission failed */

}


int recv_from_printer(int bytes) {	/* Function to read data from the printer and write it to _read_buffer */

  int n,rd=0, loop=0;
  bzero(_read_buffer, 8);

  while ( rd < bytes && loop < 10 ) {	/* Read date in a loop, in order to handle delays between transmission */

    n = usb_bulk_read(_eggprinter, _endpoint_r, (void *)_read_buffer, bytes, 50);

    if ( n == 0 ) {
      usleep(200000);
      continue;
    }

    if ( n <= 0) {
      printf("USB_BULK_READ ERROR ! Use strace for more information. \n");
      return -1;
    }

    rd += n;
    loop++;
  }

  return n;

}


int scan_usb (void) {		/* This function scans the USB and sets up the Printer communication */

  struct usb_config_descriptor *cdesc;
  struct usb_interface *interface;
  struct usb_interface_descriptor *idesc;
  struct usb_endpoint_descriptor *epdesc;

  usb_init();
  usb_find_busses();
  usb_find_devices();

  busses = usb_get_busses();

  for (bus = busses; bus; bus = bus->next) {		/* Scanning trough all Busses and Devices */
    for (dev = bus->devices; dev; dev = dev->next) {

      if (dev->descriptor.idVendor == 0x07cf) {

        if (dev->descriptor.idProduct == 0x4003 || dev->descriptor.idProduct == 0x4001) {

							/* You may want to add the ProductId for the KP-C50 like the folowing:
	                                                   (Replace 0xXXXX with the ProductId)
							   if (dev->descriptor.idProduct == 0x4003 || dev->descriptor.idProduct == 0xXXXX) {
							*/

	  cdesc = dev->config;			/* Setting all necessariy device variables */
	  interface = cdesc->interface;
	  idesc = interface->altsetting;
	  epdesc = idesc->endpoint;

	  _endpoint_w = epdesc->bEndpointAddress;	/* Writing Endpoint  */

	  epdesc = &idesc->endpoint[1];			/* Switch endpoint to get the ... */
	  _endpoint_r = epdesc->bEndpointAddress;	/* ...reading EndPoint */

	  _eggprinter = usb_open(dev);


	  if (usb_claim_interface(_eggprinter, _interfacenumber) < 0) {
	    printf("ERROR: Could not open Interface! Make sure you have the appopriate rights. \n");
	    return -1;
  	  }
	  return 0;
 	}
      }

    }
  }

 return 1;

}

int open_printer () {

  int found;

  found = scan_usb();
  printf("Scanning USB ...\n");

  if (found == 0) {
    printf("Found Casio Label Printer.\n");
     }

  if (found != 0) {
    printf("USB-Scan failed.\n\n");
     return -1;
  }

  return 0;

}


int close_printer () {			/* Closes the Printer */

  usb_release_interface(_eggprinter, _interfacenumber);
  usb_close(_eggprinter);
  printf("Printer closed. \n\n");
  return 0;
}

int display_printer_status() {

  int ret = 0;			/* For communication sheme look at the Casio Command Spec */

  ret = open_printer();

  if(ret < 0){
    return -1;
  }

  bzero(_write_buffer, 64);		/* The Writebuffer has to be cleared with zeros everytime*/
  _write_buffer[0] = ESC;		/* The Readbuffer is cleared automatically before reading */
  _write_buffer[1] = _I;

  write_to_printer(2);
  recv_from_printer(1);

  bzero(_write_buffer, 64);
  _write_buffer[0] = ENQ;

  write_to_printer(1);
  recv_from_printer(8);

  printf("Status Packet request answer: \n\n");
  printf("5: %X  Product-Type 1 = 0x01\n",_read_buffer[4]);
  printf("6: %X  Product-Type 2 =",_read_buffer[5]);

  if(_read_buffer[5] == 0x26) {
    printf(" 0x26 => 0x0126 (KLP1000)\n");
  }

  if(_read_buffer[5] == 0x27) {
    printf(" 0x27 => 0x0127 (KP-C50)\n");
  }

  if(_read_buffer[5] == 0x28) {
    printf(" 0x28 => 0x0128 (KP-C10)\n");
  }

  printf("7: %X  Version Information\n",_read_buffer[6]);
  printf("8: %X  Support Information\n\n",_read_buffer[7]);

  bzero(_write_buffer, 64);
  _write_buffer[0] = ACK;

  write_to_printer(1);
  recv_from_printer(1);

  if(_read_buffer[0] == EOT) {
    printf("Transfer complete.\n");
  }
  close_printer();
  return 0;
}


int send_raw_data() {

  int count_blocks = 0,		/* Counter for Fileposition */
      count_buffer = 0,		/* Count transferred buffers max 17 */
      count = 0,
      ready = 0,
      data_position = 0;
  int ret = 0;		/* For communication sheme look at the Casio Command Spec */
  char KLP_KPC = 0x00;

  ret = open_printer();

  if(ret < 0) {
    return -1;
  }


  bzero(_write_buffer, 64);		/* ESC_I Sequence */
  _write_buffer[0] = ESC;
  _write_buffer[1] = _I;

  write_to_printer(2);
  recv_from_printer(1);

  bzero(_write_buffer, 64);
  _write_buffer[0] = ENQ;

  write_to_printer(1);
  recv_from_printer(8);
  					/* Get Printer Type */
  KLP_KPC = _read_buffer[5];

  bzero(_write_buffer, 64);
  _write_buffer[0] = ACK;

  write_to_printer(1);
  recv_from_printer(1);

  if(_read_buffer[0] == EOT) {
    printf("Recieved EOT... continue\n");
  } else {
    printf("No EOT Recieved ! STOP \n");
    return -1;
  }

  if(KLP_KPC == 0x26) {			/* Send ESC_F to complete communication for KL-P1000*/
    printf("Printer is KL-P1000 sending ESC_F\n");

    bzero(_write_buffer, 64);		/* ESC_F Sequence */
    _write_buffer[0] = ESC;
    _write_buffer[1] = _F;

    write_to_printer(2);
    recv_from_printer(1);

    bzero(_write_buffer, 64);
    _write_buffer[0] = ENQ;

    write_to_printer(1);
    recv_from_printer(8);

    printf("Paper type = %X",_read_buffer[5]);

    if(_read_buffer[5] == 0x00) {
      printf(" => KL-P1000)\n");
    }

    if(_read_buffer[5] == 0x01) {
      printf(" => KP-C10/50)\n");
    }

    bzero(_write_buffer, 64);
    _write_buffer[0] = ACK;

    write_to_printer(1);
    recv_from_printer(1);

    if(_read_buffer[0] == EOT) {
      printf("Recieved EOT... continue\n");
    } else {
      printf("No EOT Recieved ! STOP \n");
      return -1;
    }

  }

  bzero(_write_buffer, 64);		/* ESC_P Sequence for Image formats */
  _write_buffer[0] = ESC;
  _write_buffer[1] = _P;

  printf("Send ESC_P to printer\n");

  write_to_printer(2);
  recv_from_printer(1);

  if(_read_buffer[0] == ACK) {
    printf("Recieved ACK... continue\n");
  } else {
    printf("No ACK Recieved ! STOP \n");
    return -1;
  }

  bzero(_write_buffer, 64);		/* Image-Format data for ESC_P Sequence */
  _write_buffer[0] = STX;		/* Necessary epiloge created before printing */
  _write_buffer[1] = 0x80;
  _write_buffer[2] = 0x0c;
  _write_buffer[3] = 0x00;
  _write_buffer[4] = 0x00;
  _write_buffer[5] = 0x00;
  _write_buffer[6] = 0x00;

  if (density == 1) {			/* Print Density */
    _write_buffer[7] = 0x00;
  }
  if (density == 2) {
    _write_buffer[7] = 0x01;
  }
  if (density == 3) {
    _write_buffer[7] = 0x02;
  }
  if (density == 4) {
    _write_buffer[7] = 0x03;
  }
  if (density == 5) {
    _write_buffer[7] = 0x04;
  }
  _write_buffer[8] = 0x40;		/*Height 64Dots */
  _write_buffer[9] = 0x00;


  if(KLP_KPC == 0x26) {
    _write_buffer[10] = 0x00;
    _write_buffer[11] = 0x02;
  } else {
    _write_buffer[10] = 0x02;		/* Length 512 Dots */
    _write_buffer[11] = 0x00;
  }

  _write_buffer[12] = 0x00;		/* Transfer size (64*512)/8 = 4kB */
  _write_buffer[13] = 0x00;
  _write_buffer[14] = 0x10;
  _write_buffer[15] = 0x00;

  printf("Sending Picture Information to printer !\n");

  write_to_printer(16);
  recv_from_printer(1);

  if(_read_buffer[0] == ACK) {
    printf("Recieved ACK... continue\n");
  } else {
    printf("No ACK Recieved ! STOP \n");
    return -1;
  }

  printf("Sending Picture Data-Request to printer !\n");

  bzero(_write_buffer, 64);		/* ESC_G Sequence - IMAGE DATA TRANSFER */
  _write_buffer[0] = ESC;
  _write_buffer[1] = _G;

  write_to_printer(2);
  recv_from_printer(1);

  if(_read_buffer[0] == ACK) {
    printf("Recieved ACK... continue\n");
  } else {
    printf("No ACK Recieved ! STOP \n");
    return -1;
  }

  printf("Sending Picture Data to printer !\n");

  for (count = 1; count < 5; count++) {
    bzero(_write_buffer, 64);
    _write_buffer[0] = STX;
    _write_buffer[1] = 0x00;
    _write_buffer[2] = 0x00;
    _write_buffer[3] = 0x04;

    for(count_buffer = 4; count_buffer < 64; count_buffer++) {
      _write_buffer[count_buffer] = data[(count_buffer -4) + data_position];
    }
    data_position = (count_buffer - 4) + data_position;
    write_to_printer(64);				/* 60 Bytes Data */
    bzero(_write_buffer, 64);					/* 4 Bytes Information */

    for(count_blocks = 1; count_blocks < 16 ; count_blocks++) {	/* 15 x 64 Bytes Data */

      for(count_buffer = 0; count_buffer < 64; count_buffer++) {
        _write_buffer[count_buffer] = data[count_buffer+data_position];
      }
      write_to_printer(64);
      data_position = count_buffer+data_position;

      bzero(_write_buffer, 64);

    }
    bzero(_write_buffer, 64);

    for(count_buffer = 0; count_buffer < 4; count_buffer++) {	  /* 1x 4 Bytes Data */
      _write_buffer[count_buffer] = data[data_position+count_buffer];
    }
    data_position = data_position+count_buffer;

    write_to_printer(64);					/* 4 Bytes Data 60 Bytes Zero */
    recv_from_printer(1);

    if(_read_buffer[0] != ACK) {
      printf("Transfer Error !\n");
      return -1;
    }

  }
  bzero(_write_buffer, 64);
  _write_buffer[0] = STX;
  _write_buffer[1] = 0x80;
  _write_buffer[2] = 0x80;
  _write_buffer[3] = 0x00;


  write_to_printer(64);
  bzero(_write_buffer, 64);
  for(count = 0; count < 16; count++) {
    write_to_printer(64);
  }

  recv_from_printer(1);

  printf("Sending Print Command !\n");

  bzero(_write_buffer, 64);				/* Send Print start command */
  _write_buffer[0] = ESC;
  _write_buffer[1] = _N;
  write_to_printer(2);

  recv_from_printer(1);
  if(_read_buffer[0] == ACK) {
    printf("Recieved ACK... continue\n");
  } else {
    printf("No ACK Recieved ! STOP \n");
    return -1;
  }


  if(KLP_KPC == 0x26) {					/* Send ESC_F to complete communication for KL-P1000*/

    printf("Sending ESC_T for tape feed!\n");
    printf("Waiting for Printer.");
    fflush( stdout );

    do {

      sleep(1);
      bzero(_write_buffer, 64);                             /* Send Tape feed command  for KLP-1000*/
      _write_buffer[0] = ESC;
      _write_buffer[1] = _T;
      write_to_printer(2);
      recv_from_printer(1);

      if(_read_buffer[0] == WIT) {
        printf(".");
	fflush( stdout );
      } else {

      if(_read_buffer[0] == ACK) {
        printf("\nRecieved ACK... continue\n");
        ready = 1;

        bzero(_write_buffer, 64);
        _write_buffer[0] = 0xfe;
        write_to_printer(1);
        recv_from_printer(1);

        if(_read_buffer[0] == ACK) {
          printf("Tape feed complete.\n");
        } else {
          printf("Feed error !\n");
        return -1;
        }
      }
     }
   } while (ready == 0);

  }

  ready = 0;
  printf("Waiting for printer to become ready ...\n");
  sleep(2);							/* Wait for printer to complete */
  printf("Requesting Status.");
  fflush( stdout );

  do {
    sleep(1);
    bzero(_write_buffer, 64);					/* Get status */
    _write_buffer[0] = ESC;
    _write_buffer[1] = _B;
    write_to_printer(2);
    recv_from_printer(1);
    if(_read_buffer[0] == ACK) {
      ready = 1;
      printf("\nPrinter ready.\n");
    } else {
     printf(".");
     fflush( stdout );
    }

 } while (ready == 0);

  printf("Closing printer...\n");
  close_printer();
  return 0;
}

